-- GPS virtual course for F3F script for EdgeTX and OpenTX
-- https://rc-soar.com/edgetx/lua/gpsf3f/
-- Copyright (c) Michael Shellim 2025 all rights reserved.

-- Sensor module, for polling GPS sensor telemetry
-- Tested with Radiomaster ERS-GPS sensor

local main, logging = ...  -- handle to main module passed in loadScript
local sensor = {}                   -- this module's data

local idGPS, idSats                 -- sensor ids
local lastPktStatus                 -- last packet status
local getSourceValue = getSourceValue or getValue -- fallback to getValue() for OpenTX
local lastLat, lastLon              -- cached lat/lon1. Used to detect changes and connection loss
local fmtLatLon = "%-16.11f"        -- format for logging lat/lon values

function sensor.init()
    idSats = nil
    idGPS = nil
end

-- Discover GPS and Sat sensor and cache ids
local function discoverSensors()
    -- Discover Sats telemetry
    local info = getFieldInfo("Sats")
    idSats = info and info.id or nil

    -- Discover GPS telemetry
    info = getFieldInfo("GPS")
    idGPS = info and info.id or nil
end

-- Function to fetch sensor data
-- returns  pktStatus, lat, lon
-- if error returns pktStatus, nil, nil
-- pktStatus is one of: "-", "0", "E", "C", "F"

local function fetchTelemetry()
    if not idGPS then 
        discoverSensors()
        if not idGPS then
            return "N"  -- 'No sensor'
        end
    end

    -- Sensor available. Attempt to fetch GPS packet
    local t, isCurrent, isFresh = getSourceValue(idGPS)
    if type(t) ~= 'table' or not t.lat or not t.lon then
        return "E"  -- 'Error' (GPS data not available or invalid)
    end

    -- Data returned, validate it
    if (t.lat == 0 and t.lon == 0) or not isCurrent then
        return "W"   -- 'Waiting' (not delivering data or data is not current)
    end
    -- return validated data
    return isFresh and "F" or "C", t.lat, t.lon
end

-- Main function to poll GPS telemetry. This is called from the background function.
-- returns  lat, lon if successful, else nil
local function pollGPSTelemetry()

    -- Capture and validate GPS data
    lastPktStatus, lastLat, lastLon = fetchTelemetry()

    -- Write entry to the log
    logging.write (
                getTime(),
                lastLat and string.format (fmtLatLon, lastLat) or nil,
                lastLon and string.format (fmtLatLon, lastLon) or nil,
                idSats and getSourceValue (idSats) or nil,
                lastPktStatus
            )

    return lastLat, lastLon
end

-- This function is called when in simulator mode
-- Aileron stick -> lat/lon
-- Elevator stick -> error
local function pollGPSTelemSimulated()
    
    -- Use the Ail stick to simulate sensor 
    -- Home position is 51.5074, -0.1278
    -- RH end point is 60m from home at 28°
    -- LH end point is 60m from home at 215°
    -- BearingOut should be set to 300° (via UI)
    --
    -- Use elevator stick to simulate error

    
    local t = getTime() -- timestamp for log

    -- simulate error with forward elevator
    local v = getSourceValue ("ele")
    if v > 50 then
        lastPktStatus = "E"
        logging.write (t, nil, nil, nil, lastPktStatus)
        lastLat = nil
        lastLon = nil
    else
        -- simulate lat/lon with aileron
        local latR, lonR = 51.507955837693, -0.12732516435117 -- right aileron
        local latL, lonL = 51.506884323208, -0.12838013000339 -- left aileron
        v = getSourceValue ("ail")
        lastLat = ((v + 1024) * latR + (1024 - v) * latL) / 2048
        lastLon = ((v + 1024) * lonR + (1024 - v) * lonL) / 2048
        lastPktStatus = "C"
        logging.write (t, lastLat, lastLon, nil, lastPktStatus)
        
    end
    return lastLat, lastLon
end

-- assign polling function
sensor.poll = main.isDemoMode and pollGPSTelemSimulated or pollGPSTelemetry

-- callbacks for form module
function sensor.getNumberOfSats()   return idSats and getSourceValue (idSats) or nil end
function sensor.getLastPosition()   return lastLat, lastLon end
function sensor.getPktStatus()      return lastPktStatus end

return sensor
