-- GPS virtual course for F3F script for EdgeTX and OpenTX
-- https://rc-soar.com/edgetx/lua/gpsf3f/
-- Copyright (c) Michael Shellim 2025 all rights reserved.

-- module for displaying and editing data. 
-- Includes show() function called from main loop to display data and handle key events. 

local interval, config, sensor, course, f3f, main, status, logging = ... -- module handles passed via loadScript
local form = {} -- this module's table

-- Field definitions: caption, callbacks to get, callback to adjust, lookup table, format string
local fields = {
    { caption = "Bearing",  fGet = config.getBearingOut,  fAdjust = config.adjustBearingOut },
    { caption = "Length ",  fGet = config.getCourseLen,   fAdjust = config.adjustCourseLen },
    { caption = "Base A",   fGet = config.getSideBaseA,   fAdjust = config.adjustSideBaseA, lookup = { [config.LEFT] = "Left", [config.RIGHT] = "Right" } },
    { caption = "Max legs", fGet = config.getMaxLegs,     fAdjust = config.adjustMaxLegs },
    { caption = "Start LS", fGet = config.getLsStart,    fAdjust = config.adjustLsStart },
    { caption = "Logging",  fGet = logging.isActive,      fAdjust = main.isLoggingEnabled and logging.toggle or nil},
    { caption = "Sats",     fGet = sensor.getNumberOfSats },
    { caption = "Poll (Hz)",fGet = interval.getFrequency },
    { caption = "X pos",    fGet = course.getLastX },
    { caption = "Y pos",    fGet = course.getLastY },
    { caption = "Packet",   fGet = sensor.getPktStatus },
    { caption = "Time",fGet = f3f.getLastRunTime },
}

local editableFields = {} -- table of editable fields
local fontSize          -- SMLSIZE, MIDSIZE, etc
local fontHt            -- font height for the current font size

-- state variables
local idxHighlighted    -- index of highlighted field in editableFields table
local isEditing         -- the highlighted editable field is being edited

-- Defines the layout of the form
-- Returns the height of the layout in pixels.
function form.init (xOrigin, yOrigin, width)

    -- get the font height
    -- fontSize = SMLSIZE
    fontSize = SMLSIZE
    if lcd.sizeText then
        _, fontHt = lcd.sizeText ("X", fontSize)
    else
        fontHt = 6  -- lcd.sizeText is not defined for b/w screens
    end

    local lineSpacing = 1
    local nCols = main.isWidget and 3 or 2 -- number of columns for stacked fields

    -- initialise display variables
    local dy = fontHt + lineSpacing                 -- row pitch
    local dx = math.ceil (width / nCols)            -- column pitch
    local hForm = math.ceil (#fields/nCols) * dy    -- height of form
    local yMax = yOrigin + hForm                    -- max y position for fields
    
    -- initialise state variables.
    idxHighlighted = 1
    isEditing = false

    -- initialise field stack
    local x = xOrigin
    local y = yOrigin

    -- Iterate over fields 
    for _, f in ipairs (fields)  do
        -- store the position
        f.x = x
        f.y = y

        -- advance to next position, start new column if overflow.
        y = y + dy
        if y >= yMax then
            y = yOrigin
            x = x + dx
        end

        -- if field is editable, add an entry to the editableFields table.
        if f.fAdjust then
            editableFields [#editableFields + 1] = f
        end
    end
    return (hForm) -- return height of formatted form
    
end

-- This function is called from the run() function in the main loop.
-- Processes key events and displays the form.
function form.show (event)

    -- if it's a widget and the form is not in full-screen mode, force a status message
    if main.isWidget and event == nil then
        status.setStatus("Enter full-screen mode to proceed (press long Enter)", status.HIGH_PRIORITY)
    end

    -- Form is considered 'editable' if it can process key events.
    -- Key events are received only if it's a telemetry script, or a widget in full-screen mode
    -- (after https://luadoc.edgetx.org/part_i_-_script_type_overview/widget_scripts).
    -- If form is editable, then the current field is shown with inverse/blink flags.
    local isFormEditable = not main.isWidget or event ~= nil
    if isFormEditable then
        local delta -- delta for INC/DEC key events

        if event == EVT_EXIT_BREAK then
            isEditing = false
        elseif event == EVT_VIRTUAL_ENTER then
            isEditing = not isEditing
        elseif event == EVT_VIRTUAL_INC or event == EVT_VIRTUAL_INC_REPT then
            -- supplies a delta, processed below
            delta = 1
        elseif event == EVT_VIRTUAL_DEC or event == EVT_VIRTUAL_DEC_REPT then
            delta = -1
        elseif event == EVT_VIRTUAL_MENU then
            course.menuButtonPressed()
        end

        -- inc or dec events are for field navigation and editing.
        if delta then
            if isEditing then
                -- inc/dec field value
                editableFields[idxHighlighted].fAdjust(delta)
            else
                -- or tab to next editable field.
                idxHighlighted = (idxHighlighted-1 + delta) % #editableFields + 1
            end
        end
    end

    -- iterate over fields to display them
    for _, f in ipairs (fields)  do

        -- Set display flags
        local flags = fontSize
        if isFormEditable and editableFields[idxHighlighted] == f then
            flags = flags + INVERS
            if isEditing then
                flags = flags + BLINK
            end
        end
        
        -- display the field
        local v = f.fGet()
        if v == nil then
            v = "---"
        elseif f.lookup then
            v = f.lookup [v]
        elseif f.format then
            v = string.format (f.format, v)
        end
        lcd.drawText (f.x, f.y, f.caption .. ':'.. v, flags)
    end
end

return form

