-- GPS virtual course for F3F script for EdgeTX and OpenTX
-- https://rc-soar.com/edgetx/lua/gpsf3f/
-- Copyright (c) Michael Shellim 2025 all rights reserved.

-- module for detecting course events and displaying course map
local course = {}
local sensor, config, graph, status = ... -- module handles, passed in via loadScript arguments

-- state variables
local lastx = nil -- cache last x and y positions in slope coordinates
local lasty = nil
local homeLat = nil -- home position
local homeLon = nil

-- event constants visible outside the module
course.IN_FROM_BASE_A = 1 -- A and B constants have opposite signs
course.IN_FROM_BASE_B = -1
course.OUT_FROM_BASE_A = 2
course.OUT_FROM_BASE_B = -2
course.GPS_ERROR_NO_HOME = 998
course.GPS_ERROR_NO_DATA = 999

-- forward declarations for helper functions
local geoToCartesianFast

-- Function called from f3f.background() to poll for base crossing events.
--  Using sensor and home positions (lat/lon), calculates sensor position (x,y) with
--  origin at home position and axis rotated by course bearing.
--  Tests if sensor has crossed a base by comparing current and last x-values.
--  If error found, sets the status line text (mid priority).
--  Returns 
--      event code, or course.GPS_ERROR, or nil if no event detected
function course.pollForBaseEvent()
    -- Poll the sensor for data
    local sensLat, sensLon = sensor.poll()
    if not sensLat then
        lastx = nil -- failed
        lasty = nil
        return course.GPS_ERROR_NO_DATA
    end

    -- Check we have a home position - needed to calculate x,y position
    if homeLat == nil or homeLon == nil then
        lastx = nil -- failed
        lasty = nil
        return course.GPS_ERROR_NO_HOME
    end

    -- Calculate sensor position relative to home position in Cartesian coordinates 
    -- +ve x towards RH base, +ve y is out from slope
    local bearingOut, courseLen, sideBaseA = config.getCourseConfiguration()
    local x, y = geoToCartesianFast(homeLat, homeLon, sensLat, sensLon, bearingOut)

    -- use x, y position to detect crossing of a base 
    -- assume Base A is left side, correct later

    if not lastx or not lasty then
        -- Can't compare last and current positions
        -- Update last position and bail out.
        lastx = x
        lasty = y
        return nil
    end

    local len = courseLen / 2
    local event = nil
    if x >= len and lastx < len then
        event = course.OUT_FROM_BASE_B
    elseif x <= -len and lastx > -len then
        event = course.OUT_FROM_BASE_A
    elseif x >= -len and lastx < -len then
        event = course.IN_FROM_BASE_A
    elseif x <= len and lastx > len then
        event = course.IN_FROM_BASE_B
    end

    -- Flip event if base A is RH side
    if event and sideBaseA == config.RIGHT then
        event = -event
    end

    -- update last position and return event
    lastx = x
    lasty = y
    return event
end

-- Displays a map
function course.show()
    -- draw base A and B
    local courseLenDiv2 = config.getCourseLen() / 2
    local isLeftBaseA = config.getSideBaseA() == config.LEFT
    graph.drawBlob(-courseLenDiv2, 0, isLeftBaseA)       -- LH base
    graph.drawBlob(courseLenDiv2, 0, not isLeftBaseA)    -- RH base

    -- draw home position and sensor position if defined
    if homeLat then   
        graph.drawCross(0, 0)                                -- draw home position
        if lastx then graph.drawDiagCross(lastx, lasty) end  -- draw sensor position
    else
        graph.drawSlats (0,0)
    end
end

-- Updates home position from cached lat/lon (if available)
function course.updateHomePosition()
    local lat, lon = sensor.getLastPosition()
    if lat and lon then
        homeLat = lat
        homeLon = lon
        return true
    end
end


-- =========== Helper functions ================

-- Function to convert geographic coordinates (lat/lon) to Cartesian coordinates (x,y)
-- With origin at P1 and axis rotated by theta (positive values counter-clockwise)
-- Optimized for small distances between P1 and P2
function geoToCartesianFast(lat1, lon1, lat2, lon2, theta)
    
    local earthRadius = 6371000 -- Earth radius in meters
    theta = math.rad(theta)     -- Convert theta to radians

    -- Calculate standard planar coordinates (+ve x = west to east, +ve y = south to north)
    -- This uses a simple equirectangular projection
    local dLon = (lon2 - lon1)
    local dLat = (lat2 - lat1)

    -- Calculate east-west distance
    local x = math.rad(dLon) * earthRadius * math.cos(math.rad((lat1 + lat2) / 2))

    -- Calculate north-south distance
    local y = math.rad(dLat) * earthRadius

    -- Rotate the coordinates around the origin by theta (positive theta = clockwise axis rotation)
    local cosTheta = math.cos(theta)
    local sinTheta = math.sin(theta)

    -- Apply rotation transformation
    local xRotated = x * cosTheta - y * sinTheta
    local yRotated = x * sinTheta + y * cosTheta

    return xRotated, yRotated
end

-- Info functions
-- function course.getLastPos()
--    return (lastx and lasty) and
--        (string.format("%.1f", lastx) .. ", " .. string.format("%.1f", lasty) or nil)
-- end

function course.getLastX() return (lastx) and string.format("%.1f", lastx) or nil end
function course.getLastY() return (lasty) and string.format("%.1f", lasty) or nil end


function course.isHomeset()  return homeLat ~= nil and homeLon ~= nil end

return course
