-- GPS virtual course for F3F script for EdgeTX and OpenTX
-- https://rc-soar.com/edgetx/lua/gpsf3f/
-- Copyright (c) Michael Shellim 2025 all rights reserved.

--[[ 
Event/state table
+---------------+------------------+------------------+-------------------+------------------+-------------------+
|        STATE  | Waiting for conn | Waiting for Home | Waiting for Start | InClimbout       | OnCourse          |
| EVENT         | (initial state)  |                  |                   |                  |                   |
+---------------+------------------+------------------+-------------------+------------------+-------------------+
| GPS lost      | ---              | Waiting for conn | Waiting for conn  | Waiting for conn | Waiting for conn  |
+---------------+------------------+------------------+-------------------+------------------+-------------------+
| GPS received  | If Home is set   | ---              | ---               | ---              | ---               |
|               | Wait Start else  |                  |                   |                  |                   |
|               | Waiting for Home |                  |                   |                  |                   |
+---------------+------------------+------------------+-------------------+------------------+-------------------+
| Home button   | ---              | {update home pos}| {update home pos} | {update home pos}| {update home po}  |
+---------------+------------------+------------------+-------------------+------------------+-------------------+
| Start button  | ---              | {"home not set"} | In Climbout       | In Climbout      | In Climbout       |
+---------------+------------------+------------------+-------------------+------------------+-------------------+
| Abort button  | ---              | ---              | ---               | Waiting for Start| Waiting for Start |
+---------------+------------------+------------------+-------------------+------------------+-------------------+
| Passed base A | ---              | ---              | ---               | On course        | ---               |
| into course   |                  |                  |                   |                  |                   |
+---------------+------------------+------------------+-------------------+------------------+-------------------+
| Passed base B | ---              | ---              | ---               | ---              | ---             |
| out of course |                  |                  |                   |                  |                   |
+---------------+------------------+------------------+-------------------+------------------+-------------------+
| Passed base A | ---              | ---              | ---               | ---              | if run done then  |
| out of course |                  |                  |                   |                  | waiting for Start |
+---------------+------------------+------------------+-------------------+------------------+-------------------+
 ]]
-- module for handling external switch events and managing F3F run logic.
local f3f = {}
local course, timer, switch, config, status = ... -- module handles passed by loadScript()

-- module variables
local startSwitch            -- logical switch number to start (on) or abort (off) run
local menuButtonPressed     -- true if menu button pressed

local MAX_CLIMBOUT_SECS = 30

-- module variables

local legsCompleted
local lastRunTime
local nextBase
local t0        -- base time for run timer (10 mS units)
local callSecs = {
    [30] = true,
    [20] = true,
    [15] = true,
    [10] = true,
    [9] = true,
    [8] = true,
    [7] = true,
    [6] = true,
    [5] = true,
    [4] = true,
    [3] = true,
    [2] = true,
    [1] = true,
}

local state

local WAITING_CONNECTION = 1
local WAITING_HOME = 2
local WAITING_START = 3
local CLIMBOUT = 4
local ON_COURSE = 5

local statusText = {
    [WAITING_CONNECTION]= "Waiting for GPS data",
    [WAITING_HOME]      = "Set center to proceed",
    [WAITING_START]     = "Press Start when ready",
    [CLIMBOUT]          = "In climbout",
    [ON_COURSE]         = function () return "On course " .. legsCompleted .. " completed" end,
    }


-- forward declarations
local transitionTo, beep

-- intialise this module
function f3f.init()
    -- create virtual switches to start and stop the run
    startSwitch = switch:new({ getlogSwitch = config.getLsStart })
    menuButtonPressed = false
    
    -- inititalise state 
    lastRunTime = nil
    transitionTo(WAITING_CONNECTION)
end


-- Background function
-- Polls sensor, switches and timer and manages state transitions
-- Status text at this level is low priority.
function f3f.background()

    -- set the status line
    local s = statusText [state]
    if type(s) == "function" then s = s () end
    status.setStatus (s)

    -- Check if home update requested (Menu button pressed)
    local isHomeUpdateRequest = menuButtonPressed
    menuButtonPressed = false

    -- poll for base crossing events
    local courseEvent = course.pollForBaseEvent()

       -- Handle GPS data/errors
    if courseEvent == course.GPS_ERROR_NO_DATA then
        if state ~= WAITING_CONNECTION then
            playFile ("gf3dis.wav")  -- no data
            transitionTo(WAITING_CONNECTION)
        end
        return
    else
        -- We have GPS data
        -- If we were waiting for connection, move to next state
        if state == WAITING_CONNECTION then
            playFile ("gf3con.wav")  -- connection established
            if course.isHomeset() then
                transitionTo(WAITING_START)
            else
                transitionTo(WAITING_HOME)
            end
            return
        end
    end


    -- --------------------
    -- Handle button events
    -- --------------------

    -- Home switch pressed?
    if isHomeUpdateRequest then
        if course.updateHomePosition() then
            playFile ("gf3ctr.wav") -- 'centre registered'
            if state == WAITING_HOME then
                transitionTo(WAITING_START)
                return
            end
        else
            playFile ("gf3noc.wav") -- failed to update home position
        end
    end

    -- Start/abort switch event
    local startSwitchEvent = startSwitch:event()
    if startSwitchEvent == switch.PRESSED then
        -- Start requested
        if state == WAITING_HOME or state == WAITING_CONNECTION then
            playFile ("gf3cen.wav") -- centre not set
        elseif state ~= CLIMBOUT then
            transitionTo(CLIMBOUT)
            return
        end
    elseif startSwitchEvent == switch.RELEASED then
        -- Abort requested
        if state > WAITING_START then
            playFile ("gf3abt.wav") -- run aborted
            transitionTo(WAITING_START)
            return
        end
    end

    -- -------------------
    -- Handle base and timer events. 
    -- These only apply to CLIMBOUT and ON_COURSE states
    -- --------------------
    if state == CLIMBOUT then

        -- handle base events
        if courseEvent == course.IN_FROM_BASE_A then
            transitionTo(ON_COURSE)
            return
        elseif courseEvent == course.OUT_FROM_BASE_A then
            playFile("gf3off.wav") -- "off course"
        end

        -- Call the climbout countdown
        local ticksRemaining = timer.poll()
        if ticksRemaining then
            -- A tick event has occurred. 
            if ticksRemaining == 0 then
                -- Climbout time expired, start the run timer
                t0 = getTime()  -- save base time for run timer
                beep()
            elseif callSecs[ticksRemaining] then
                -- Climbout time has not expired and we're on a callout
                playNumber(ticksRemaining, (ticksRemaining > 10) and UNIT_SECONDS or UNIT_RAW)
            end
        end
        
    elseif state == ON_COURSE then
        if courseEvent == nextBase then
            legsCompleted = legsCompleted + 1
            if legsCompleted == config.getMaxLegs() then
                -- Final leg completed
                beep(3)                                         -- Signal end of run
                lastRunTime = getTime() - t0                    -- run time in 1/100'ths of seconds
                playNumber(lastRunTime + 5, UNIT_SECONDS, PREC2) -- Call out run time (plays one decimal digit, so round it.)
                lastRunTime = lastRunTime/100 -- seconds
                transitionTo(WAITING_START)
                return
            else
                -- non-final leg completed
                -- call out leg number and flip next base A <-> B
                beep()
                -- system.playNumber(legsCompleted)
                playNumber(legsCompleted, 0)
                nextBase = -nextBase
            end
        end
    end

end

-- This function called in form module to indicate that the Menu button has been pressed
-- Menu button is used to update the home position
function course.menuButtonPressed()
    menuButtonPressed = true
end

-- Callback functions for form module
function f3f.getLastRunTime() return lastRunTime end
-- function f3f.getState() return stateNames[state] end

-- ==== Local helper functions (require forward declarations) ===

-- Transition to a new state
function transitionTo(newState)
    if newState == CLIMBOUT then
        playFile("gf3lau.wav")
        timer.start(MAX_CLIMBOUT_SECS)  -- start the climbout timer
        t0 = nil                        -- reset base time for run timer
        
    elseif newState == ON_COURSE then
        -- if run timer wasn't started at expiry of climbout timer, start it now
        if not t0 then t0 = getTime()  end
        beep()
        playFile("gf3onc.wav")
        nextBase = course.OUT_FROM_BASE_B
        legsCompleted = 0
    end
    state = newState
end

function beep(j)
    j = j or 1
    for _ = 1, j do
        playTone(2550, 160, 40, 0, 10) -- play tone, use radio settings volume
    end
end

return f3f
