-- GPS virtual course for F3F script for EdgeTX and OpenTX
-- https://rc-soar.com/edgetx/lua/gpsf3f/
-- Copyright (c) Michael Shellim 2025 all rights reserved.

-- Sensor module, for polling GPS sensor telemetry
-- Tested with Radiomaster ERS-GPS sensor

local main, counter, logging = ...  -- handle to main module passed in loadScript
local sensor = {}                   -- this module's data

local idGPS, idSats                 -- sensor ids
local lastPktStatus                 -- last packet status
local pollCounter = counter:new()   -- counter of poll attempts
local getSourceValue = getSourceValue or getValue -- fallback to getValue() for OpenTX
local lastLat, lastLon              -- cached lat/lon1. Used to detect changes and connection loss
local fmtLatLon = "%-16.11f"        -- format for logging lat/lon values

-- Discover GPS and Sat sensor and cache ids
function sensor.init()
    -- Discover Sats telemetry
    local info = getFieldInfo("Sats")
    idSats = info and info.id or nil

    -- Discover GPS telemetry
    info = getFieldInfo("GPS")
    idGPS = info and info.id or nil
end

-- Function to fetch sensor data
-- returns  pktStatus, lat, lon
-- if error returns pktStatus, nil, nil
-- pktStatus is one of: "-", "0", "E", "C", "F"

local function fetchTelemetry()
    -- Sensor found?
    if not idGPS then
        return "N"  -- 'No sensor'
    end

    -- Attempt to fetch GPS packet
    local t, isCurrent, isFresh = getSourceValue(idGPS)
    if type(t) ~= 'table' or not t.lat or not t.lon then
        return "E"  -- 'Error' (GPS data not available or invalid)
    end

    -- Data returned, validate it
    if (t.lat == 0 and t.lon == 0) or not isCurrent then
        return "W"   -- 'Waiting' (not delivering data or data is not current)
    end
    -- return validated data
    return isFresh and "F" or "C", t.lat, t.lon
end

-- Main function to poll GPS telemetry. This is called from the background function.
-- returns  lat, lon if successful, else nil
local function pollGPSTelemetry()

    -- Increment call count
    pollCounter:increment() -- increment call counter

    -- Capture and validate GPS data
    local pktStatus, lat, lon = fetchTelemetry()

    -- Call alert if sensor lost or regained. Non-nil lat indicates presence of good data.
    if lat then 
        if not lastLat then 
            playFile ("gf3con.wav") -- sensor regained
        end
    elseif lastLat then 
        playFile ("gf3dis.wav")     -- sensor lost
    end

    -- Write entry to the log
    logging.write (
                getTime(),
                lat and string.format (fmtLatLon, lat) or nil,
                lon and string.format (fmtLatLon, lon) or nil,
                idSats and getSourceValue (idSats) or nil,
                pktStatus
            )


    -- rediscover sensors if no data received.
    if not lat then
        sensor.init()
    end
    
    -- cache lat, lon and packet status 
    lastLat = lat
    lastLon = lon
    lastPktStatus = pktStatus

    return lastLat, lastLon
end

-- This function is called when in simulator mode
-- Aileron stick -> lat/lon
-- Elevator stick -> error
local function pollGPSTelemSimulated()
    pollCounter:increment()

    -- Use the Ail stick to simulate sensor 
    -- Home position is 51.5074, -0.1278
    -- RH end point is 60m from home at 28°
    -- LH end point is 60m from home at 215°
    -- BearingOut should be set to 300° (via UI)
    --
    -- Use elevator stick to simulate error

    
    local t = getTime() -- timestamp for log

    -- simulate error with forward elevator
    local v = getSourceValue ("ele")
    if v > 50 then
        if lastLat then 
            playFile ("gf3dis.wav") -- 'sensor' lost
        end
        lastPktStatus = "E"
        logging.write (t, nil, nil, nil, lastPktStatus)
        lastLat = nil
        lastLon = nil
    else
        -- simulate lat/lon with aileron
        if not lastLat then 
            playFile ("gf3con.wav") -- 'sensor' regained
        end
        local latR, lonR = 51.507955837693, -0.12732516435117 -- right aileron
        local latL, lonL = 51.506884323208, -0.12838013000339 -- left aileron
        v = getSourceValue ("ail")
        lastLat = ((v + 1024) * latR + (1024 - v) * latL) / 2048
        lastLon = ((v + 1024) * lonR + (1024 - v) * lonL) / 2048
        lastPktStatus = "C"
        logging.write (t, lastLat, lastLon, nil, lastPktStatus)
        
    end
    return lastLat, lastLon
end

-- assign polling function
sensor.poll = main.isSimulateSensor and pollGPSTelemSimulated or pollGPSTelemetry

-- callbacks for form module
function sensor.getNumberOfSats()   return idSats and getSourceValue (idSats) or nil end
function sensor.getLastPosition()   return lastLat, lastLon end
function sensor.getPollCount()      return pollCounter:getCount() end
function sensor.getPktStatus()      return lastPktStatus end

return sensor
