-- GPS virtual course for F3F script for EdgeTX and OpenTX
-- https://rc-soar.com/edgetx/lua/gpsf3f/
-- Copyright (c) Michael Shellim 2025 all rights reserved.

local version = "1.1"

-- Change Log
-- 2025-10-15 v1.1:     added sound for 'centre registered', modified sensor lost/found sounds.
-- 2025-06-15 v1.0:     refactored sensor.lua, changed sensor status codes 
-- 2025-05-06 beta 3:   longer captions for form fields
--                      improved error handling with extra voice alerts
--                      fixed timer bug
--                      forms are 2 or 3 cols depending on pane size
--                      refactored code in several modules
-- 2025-05-03 beta 2:   improved status line messages
--                      removed version from widget name


local main  = {} -- this module 
main.isWidget = ...                     -- true if widget, false if telemetry script. Passed from loader.
main.isLoggingEnabled = false           -- set true to enable logging. Referenced in f3f module. if enabled make sure to disable system logging.
main.isSimulateSensor = false           -- set true if running in sim or TX/demo mode. Referenced in sensor module.
main.path = "/scripts/telemetry/f3f/"   -- path of lua files

local options = {}  -- widget options
local runCounter    -- counter for run() function
local bgCounter     -- counter for background() function

-- Metrics for form module
function main.getBgRate () return bgCounter:getCount() end
function main.getRunRate () return runCounter:getCount() end

-- Load modules in dependency order.
local counter = loadScript (main.path .. "counter.lua")()
local status =  loadScript (main.path .. "status.lua")()
local logging = loadScript (main.path .. "logging.lua")(main)
local sensor =  loadScript (main.path .. "sensor.lua")(main, counter, logging)
local graph =   loadScript (main.path .. "graph.lua") ()
local config =  loadScript (main.path .. "config.lua")(main)
local course =  loadScript (main.path .. "course.lua")(sensor, config, graph, status)
local timer =   loadScript (main.path .. "timer.lua") ()
local switch =  loadScript (main.path .. "switch.lua")()
local f3f =     loadScript (main.path .. "f3f.lua")(course, timer, switch, config, status)
local form =    loadScript (main.path .. "form.lua")(config, sensor, course, f3f, main, status, logging)


-- initialises modules which depend on display area
local function initDisplay (x0, y0, w, h)

    -- apply padding inside the display rectangle
    local pad = (w >= 480 and h >= 272)  and 20 or 1
    x0 = x0 + pad
    y0 = y0 + pad
    w = w - 2*pad
    h = h - 2*pad

    -- intialise the three regions of the screen (form, status, graphics)
    local hForm = form.init(x0, y0, w)              -- form is always at top of display areas
    local hStatus = status.init(x0, y0, w, h, version)       -- status line is always at bottom of display areas
    graph.init (x0, y0 + hForm, w, h-hForm-hStatus) -- between form and status
end

-- Initialises remaining modules 
local function initTheRest ()
    sensor.init()
    config.init()
    f3f.init()
    runCounter = counter:new()
    bgCounter = counter:new()
end

-- Performs background processing.
--  For telemetry scripts: called by the o/s
--  For widgets: called by the o/s, and from refresh()
local function background ()
    f3f.background()
    bgCounter:increment()
end

-- Paints the display.
-- for telemetry scripts: called by the o/s 
-- for widgets: called by refresh() 
local function run(event)
    lcd.clear ()        -- required for mono screens, and (optionally) for colour screens to clear (distracting!) graphics
    form.show(event)    -- display form panel and handle key events
    course.show ()      -- display course map
    status.show()       -- display status line
    runCounter:increment()
end

if main.isWidget then

    -- ============ Functions required if running as widget ============

    -- called by the o/s to create a widget instance
    local function create (zone, options)
        initDisplay(zone.x, zone.y, zone.w, zone.h)
        initTheRest()
        local widget = {zone=zone, options=options}
        return widget
    end

    -- called when widget options change and/or the zone changes
    local function update(widget, options)
        widget.options = options
        local zone = widget.zone
        initDisplay(zone.x, zone.y, zone.w, zone.h) -- reinitialise display
    end

    -- Runs periodically when widget is visible
    local function refresh(widget, event, touchState)
        background()    -- keep responsive
        run(event)      -- show the form and handle key events
    end

    return {name="gpsf3f", options=options, create=create, update=update, refresh=refresh, background=background }

else

    -- ============ Functions required if running as telemetry script  ============

    -- intialisations for telemetry script
    local function initTelemetryScript()
        initDisplay (0, 0, LCD_W, LCD_H) -- full screen
        initTheRest()
    end

    return {run = run, init = initTelemetryScript, background = background}
end
