-- GPS virtual course for F3F script for EdgeTX and OpenTX
-- https://rc-soar.com/edgetx/lua/gpsf3f/
-- Copyright (c) Michael Shellim 2025 all rights reserved.

-- module for handling external switch events and managing F3F run logic.
local f3f = {}
local course, timer, switch, config, status = ... -- module handles passed by loadScript()

-- module variables
local swStart            -- logical switch number to start run
local swStop             -- logical switch number to abort run

local MAX_CLIMBOUT_SECS = 30

-- state variables and constants
local state
local IDLE = 1
local CLIMBOUT = 2
local ON_COURSE = 3
local stateNames = { "IDLE", "CLMB", "ONCRS" }

local legsCompleted
local lastRunTime
local nextBase
local t0        -- base time for run timer (10 mS units)
local callSecs = {
    [30] = true,
    [20] = true,
    [15] = true,
    [10] = true,
    [9] = true,
    [8] = true,
    [7] = true,
    [6] = true,
    [5] = true,
    [4] = true,
    [3] = true,
    [2] = true,
    [1] = true,
}

-- forward declarations
local transitionTo, beep

-- intialise this module
function f3f.init()
    -- create virtual switches to start and stop the run
    swStart = switch:new({ getlogSwitch = config.getLsStart })
    swStop = switch:new({ getlogSwitch = config.getLsStop })

    -- inititalise state 
    lastRunTime = nil
    transitionTo(IDLE)
end


-- Background function
-- Polls sensor, switches and timer and manages state transitions
-- Status text at this level is low priority.
function f3f.background()

    -- clear the status line
    status.reset()

    -- poll for base crossing events
    local baseEvent = course.pollForBaseEvent()

    -- Abort to IDLE state if error or 'stop' switch pressed
    if baseEvent == course.GPS_ERROR or swStop:isPressed()then
        if state ~= IDLE then
            playFile("gf3abt.wav") -- "run aborted"
            transitionTo(IDLE)
        end
        return
    end

    -- Transition to CLIMBOUT if 'start' switch pressed
    if swStart:isPressed() then transitionTo(CLIMBOUT) return end

    -- We have good GPS data
    -- Update state depending on the current state and the base crossing event
    if state == CLIMBOUT then

        -- handle base events
        if baseEvent == course.IN_FROM_BASE_A then
            transitionTo(ON_COURSE)
            return
        elseif baseEvent == course.OUT_FROM_BASE_A then
            playFile("gf3off.wav") -- "off course"
        end

        -- Call the countdown
        local ticksRemaining = timer.poll()
        if ticksRemaining then
            -- A tick event has occurred. 
            if ticksRemaining == 0 then
                -- Climbout time expired, start the run timer
                t0 = getTime()  -- save base time for run timer
                beep()          -- gentle reminder that the run timer has started
            elseif callSecs[ticksRemaining] then
                -- call out the ticks remaining
                playNumber(ticksRemaining, (ticksRemaining > 10) and UNIT_SECONDS or UNIT_RAW)
            end
        end

        -- Set the status line text
        status.setStatus ("In climbout", status.LOW_PRIORITY)
        
    elseif state == ON_COURSE then
        if baseEvent == nextBase then
            legsCompleted = legsCompleted + 1
            if legsCompleted == config.getMaxLegs() then
                -- Final leg completed
                beep(3)                                         -- Signal end of run
                lastRunTime = getTime() - t0                    -- run time in 10 mS units
                playNumber(lastRunTime, UNIT_SECONDS, PREC2)    -- Call out run time (plays one decimal digit. Bug?)
                transitionTo(IDLE)
                return
            else
                -- non-final leg completed
                -- call out leg number and flip next base A <-> B
                beep()
                playNumber(legsCompleted, 0)
                nextBase = -nextBase
            end
        end
        status.setStatus ("On course, " .. legsCompleted .. " completed", status.LOW_PRIORITY)
    else -- must be IDLE state
        status.setStatus ("Ready", status.LOW_PRIORITY)
    end
end

-- Callback functions for form module
function f3f.getLastRunTime() return lastRunTime end
function f3f.getState() return stateNames[state] end

-- ==== Local helper functions (require forward declarations) ===

-- Transition to a new state
function transitionTo(newState)
    if newState == CLIMBOUT then
        playFile("gf3lau.wav")
        timer.start(MAX_CLIMBOUT_SECS)  -- start the climbout timer
        t0 = nil                        -- reset base time for run timer
        
    elseif newState == ON_COURSE then
        -- if run timer wasn't started at expiry of climbout timer, start it now
        if not t0 then t0 = getTime()  end
        beep()
        playFile("gf3onc.wav")
        nextBase = course.OUT_FROM_BASE_B
        legsCompleted = 0
    end
    state = newState
end

function beep(j)
    j = j or 1
    for _ = 1, j do
        playTone(2550, 160, 40, 0, 10) -- play tone, use radio settings volume
    end
end

return f3f
