-- GPS virtual course for F3F script for EdgeTX and OpenTX
-- https://rc-soar.com/edgetx/lua/gpsf3f/
-- Copyright (c) Michael Shellim 2025 all rights reserved.

-- Handles display of status line
local status = {}

local font = SMLSIZE    -- font for status text
local hStatus           -- height of status text
local x0, y0, w, h      -- bounding rectangle
local version

-- state variables
local currString         -- status string to display
local currPriority       -- priority

-- Externally visible constants
status.LOW_PRIORITY = 1   -- for F3F status messages
status.MID_PRIORITY = 2   -- for GPS connection errors
status.HIGH_PRIORITY = 3  -- for 'enter full screen mode' messages

-- intialise the status line.
-- returns the height of the status line
function status.init(...)
    x0, y0, w, h, version = ...

    -- Set line height
    if lcd.sizeText then
        _, hStatus = lcd.sizeText ("X", font)
    else
        hStatus = 6  -- use a default value, corresponds to SML_SIZE font.
    end
    hStatus = hStatus + 1   -- add 1 pixel for line spacing

    currString = ""
    currPriority = status.LOW_PRIORITY
    return hStatus  -- return the height of the status line
end

-- Display status string and script version.
function status.show()
    local y = y0 + h - hStatus
    lcd.drawText(x0,        y, currString, font)
    lcd.drawText(x0 + w,    y, "v" .. version, font + RIGHT)
end

-- Reset the status line. Call this at the start of the background loop.
function status.reset()
    currString = ""
    currPriority = status.LOW_PRIORITY
end

-- set the status string to be displayed. 
function status.setStatus(newString, newPriority)
    newPriority = newPriority or status.LOW_PRIORITY
    if newPriority >= currPriority then
        currString = newString
        currPriority = newPriority
    end
end

return status
