-- GPS virtual course for F3F script for EdgeTX and OpenTX
-- https://rc-soar.com/edgetx/lua/gpsf3f/
-- Copyright (c) Michael Shellim 2025 all rights reserved.

-- module for managing course configuration
local config={}
local main = ... -- module handles passed by loadScript

-- exposed constants
config.LEFT = 0
config.RIGHT = 1

-- module variables
local bearingOut -- bearing perpendicular to course into wind.
local courseLen  -- length of course (meters)
local sideBaseA  -- side of base A Left/Right
local lsStart    -- logical switch number for start, false->true transition to start
local lsStop     -- logical switch number to terminate, false->true transition to terminate
local maxLegs    -- number of legs to fly (default 10)

-- configuration file name
local configFile = main.path .. "config.ini"

-- forward declarations
local minmax, writeConfig

function config.init ()
    -- open config file
    local f = io.open(configFile, "r")

    -- local function to read a config parameter. Returns number or default value
    local function readNumber(default) 
        return f and tonumber((io.read (f, 5))) or default
    end

    -- read config values
    bearingOut = readNumber(0)
    courseLen  = readNumber(100)
    sideBaseA  = readNumber(config.LEFT)
    lsStart    = readNumber(1)
    lsStop     = readNumber(2)
    maxLegs    = readNumber(10)

    -- Close the config file
    if f then io.close (f) end
end

-- Return the current configuration
function config.getCourseConfiguration()
    return bearingOut, courseLen, sideBaseA
end

-- ======== functions called in form module
function config.getBearingOut () return bearingOut end
function config.getCourseLen() return courseLen end
function config.getSideBaseA() return sideBaseA end
function config.getLsStart() return lsStart end
function config.getLsStop() return lsStop end
function config.getMaxLegs() return maxLegs end

function config.adjustBearingOut (sign) bearingOut = (bearingOut + 5 * sign + 360) % 360; writeConfig() end
function config.adjustCourseLen (sign) courseLen = minmax (courseLen + 10*sign, 10,100); writeConfig() end
function config.adjustSideBaseA(...) sideBaseA = 1 - sideBaseA; writeConfig() end
function config.adjustLsStart(sign) lsStart = minmax (lsStart + sign, 1,64); writeConfig() end
function config.adjustLsStop(sign) lsStop = minmax (lsStop + sign, 1,64); writeConfig() end
function config.adjustMaxLegs(sign) maxLegs = minmax (maxLegs + 5 * sign, 5, 100); writeConfig() end

-- ========== helper functions (require forward declarations)

function writeConfig ()
    local f = io.open(configFile, "w")
    if not f then return end

    -- local function to append a number to str
    local str = ""
    local function append (v) str = str .. string.format ("%5i", v) end

    -- Append config values to string
    append (bearingOut)
    append (courseLen)
    append (sideBaseA)
    append (lsStart)
    append (lsStop)
    append (maxLegs)

    -- write string to file and close it
    io.write(f, str)
    io.close(f)
end

function minmax (v, min, max)
    if v < min then
        v = min
    elseif v > max then
        v = max
    end
    return v
end

return config